/*
 * File:		cpu/MCF5200/MCF5307.h
 * Purpose:		MCF5307 definitions
 *
 * Notes:		This file automatically included.
 *
 * Author:		Michael Norman
 * Date:		3-11-99
 * 
 * Modifications:
 *		
 *		User: Jeffri Tan		Date:4/19/2000 
 *		- Made modifications to compile under Metrowerks 
 *		  Codewarrior E68k.
 */

#ifndef _CPU_MCF5307_H
#define _CPU_MCF5307_H

#ifdef __MWERKS__
	/* Turn off C++ so functions can call assembly routines */
	#pragma cplusplus off
	/* Turn off optimizations to prevent inadvertent optimizing of critical code */
	#pragma optimization_level 0

	#define M5307C3			// Example stationery on the M5307C3
	#define MCF5307A
#endif

/***********************************************************************/
/*
 * Misc. Defines
 */

#ifdef	FALSE
#undef	FALSE
#endif
#define FALSE	(0)

#ifdef	TRUE
#undef	TRUE
#endif
#define	TRUE	(1)

//#ifdef	NULL
//#undef	NULL
//#endif
//#define NULL	(0)

/***********************************************************************/
/*
 * The basic data types
 */

typedef unsigned char		uint8;  /*  8 bits */
typedef unsigned short int	uint16; /* 16 bits */
typedef unsigned long int	uint32; /* 32 bits */

typedef signed char			int8;   /*  8 bits */
typedef signed short int	int16;  /* 16 bits */
typedef signed long int		int32;  /* 32 bits */

/***********************************************************************/
/*
 * Common M68K & ColdFire definitions
 */

#define ADDRESS			uint32
#define INSTRUCTION		uint16
#define ILLEGAL			0x4AFC
#define CPU_WORD_SIZE	16

/***********************************************************************/

/*
 * Routines and macros for accessing Input/Output devices
 */

#define cpu_iord_8(ADDR)        *((volatile uint8 *)(ADDR))
#define cpu_iord_16(ADDR)       *((volatile uint16 *)(ADDR))
#define cpu_iord_32(ADDR)       *((volatile uint32 *)(ADDR))

#define cpu_iowr_8(ADDR,DATA)   *((volatile uint8 *)(ADDR)) = (DATA)
#define cpu_iowr_16(ADDR,DATA)  *((volatile uint16 *)(ADDR)) = (DATA)
#define cpu_iowr_32(ADDR,DATA)  *((volatile uint32 *)(ADDR)) = (DATA)

/***********************************************************************/

#define MCF5200_SR_T		(0x8000)
#define MCF5200_SR_S		(0x2000)
#define MCF5200_SR_M		(0x1000)
#define MCF5200_SR_IPL		(0x0700)
#define MCF5200_SR_IPL_0	(0x0000)
#define MCF5200_SR_IPL_1	(0x0100)
#define MCF5200_SR_IPL_2	(0x0200)
#define MCF5200_SR_IPL_3	(0x0300)
#define MCF5200_SR_IPL_4	(0x0400)
#define MCF5200_SR_IPL_5	(0x0500)
#define MCF5200_SR_IPL_6	(0x0600)
#define MCF5200_SR_IPL_7	(0x0700)
#define MCF5200_SR_X		(0x0010)
#define MCF5200_SR_N		(0x0008)
#define MCF5200_SR_Z		(0x0004)
#define MCF5200_SR_V		(0x0002)
#define MCF5200_SR_C		(0x0001)

/***********************************************************************/

/*
 * The ColdFire family of processors has a simplified exception stack
 * frame that looks like the following:
 *
 *              3322222222221111 111111
 *              1098765432109876 5432109876543210
 *           8 +----------------+----------------+
 *             |         Program Counter         |
 *           4 +----------------+----------------+
 *             |FS/Fmt/Vector/FS|      SR        |
 *   SP -->  0 +----------------+----------------+
 *
 * The stack self-aligns to a 4-byte boundary at an exception, with
 * the FS/Fmt/Vector/FS field indicating the size of the adjustment
 * (SP += 0,1,2,3 bytes).
 */

#define MCF5200_RD_SF_FORMAT(PTR)	\
	((*((uint16 *)(PTR)) >> 12) & 0x00FF)

#define MCF5200_RD_SF_VECTOR(PTR)	\
	((*((uint16 *)(PTR)) >>  2) & 0x00FF)

#define MCF5200_RD_SF_FS(PTR)		\
	( ((*((uint16 *)(PTR)) & 0x0C00) >> 8) | (*((uint16 *)(PTR)) & 0x0003) )

#define MCF5200_SF_SR(PTR)	*((uint16 *)(PTR)+1)
#define MCF5200_SF_PC(PTR)	*((uint32 *)(PTR)+1)

#if 0
typedef struct
{
	uint16	SR;
	uint16	FS_FMT_VECTOR_FS;
	uint32	PC;
} MCF5200_STACK_FRAME;
#endif

/*********************************************************************
 *	Function Prototypes
 *********************************************************************/

void
mcf5307_wr_vbr (uint32);

void
mcf5307_wr_cacr (uint32);

void
mcf5307_wr_acr0 (uint32);

void
mcf5307_wr_acr1 (uint32);

void
mcf5307_wr_rambar (uint32);

void
mcf5307_wr_mbar (uint32);

/**********************************************************************/
/*	Macro for computing address of on-chip peripheral registers		  */
/**********************************************************************/
 
#define Mcf5307_addr(IMMP,OFFSET)	((void *)&((uint8 *)IMMP)[OFFSET])

/**********************************************************************/
/*	Macros for accessing the on-chip I/O resources					  */
/**********************************************************************/
 
#define Mcf5307_iord(IMMP,OFFSET,SIZE)		\
    *(volatile uint ## SIZE *)(Mcf5307_addr(IMMP,OFFSET))

#define Mcf5307_iowr(IMMP,OFFSET,SIZE,DATA)  \
    *(volatile uint ## SIZE *)(Mcf5307_addr(IMMP,OFFSET)) = (DATA)

/**********************************************************************/
/*	Cache and Access Control Register routines and defintions         */
/**********************************************************************/
 
#define MCF5307_CACR_EC		(0x80000000)		/* Cache Enable */
#define MCF5307_CACR_ESB	(0x20000000)		/* Enable Store Buffer */		    
#define MCF5307_CACR_DPI	(0x10000000)		/* Disable CPUSHL Invalidation */ 	
#define MCF5307_CACR_HLCK	(0x08000000)		/* 1/2 Cache Lock Mode */		    
#define MCF5307_CACR_CINVA	(0x01000000)		/* Cache Invalidate All */ 	    
#define MCF5307_CACR_DNFB	(0x00000400)		/* Default Noncacheable Fill Buffer */ 
#define MCF5307_CACR_DCM_00	(0x00000000)		/* Cacheable, writethrough */	    
#define MCF5307_CACR_DCM_01	(0x00000100)		/* Cacheable, copyback */		    
#define MCF5307_CACR_DCM_10	(0x00000200)		/* Cache-Inhibited, precise */    
#define MCF5307_CACR_DCM_11	(0x00000300)		/* Cache-Inhibited, imprecise */   
#define MCF5307_CACR_DW		(0x00000020)		/* Default Write Protect */	    

#define MCF5307_ACR_BASE(a)	((a)&0xFF000000)	/* Address Base */				
#define MCF5307_ACR_MASK(a)	(((a)&0xFF000000)>>8)	/* Address Mask */		    
#define MCF5307_ACR_E		(0x00008000)		/* Enable */						
#define MCF5307_ACR_S_USER	(0x00000000)		/* User Mode Access */				
#define MCF5307_ACR_S_SUPER	(0x00002000)		/* Supervisor Mode Access */	    
#define MCF5307_ACR_S_IGNORE	(0x00004000)	/* Ignore Mode Access */	    
#define MCF5307_ACR_CM_00	(0x00000000)		/* Cacheable, writethrough */	    
#define MCF5307_ACR_CM_01	(0x00000020)		/* Cacheable, copyback */		    
#define MCF5307_ACR_CM_10	(0x00000040)		/* Cache-Inhibited, precise */	    
#define MCF5307_ACR_CM_11	(0x00000060)		/* Cache-Inhibited, imprecise */	
#define MCF5307_ACR_WP		(0x00000004)		/* Write Protect */				

#define MCF5307_VBR_BASE(a)	((a)&0xFFF00000)	/* Vector Base Address */	    

#define MCF5307_RAMBAR_BASE(a)	((a)&0xFFFFFE00)	/* Base Address */		    
#define MCF5307_RAMBAR_WP	(0x00000100)		/* Write Protect */		    
#define MCF5307_RAMBAR_AS_CPU	(0x00000020)	/* CPU Space/Interrupt acknowledge */  
#define MCF5307_RAMBAR_AS_SC	(0x00000010)	/* Supervisor Code */		    
#define MCF5307_RAMBAR_AS_SD	(0x00000008)	/* Supervisor Data */		    
#define MCF5307_RAMBAR_AS_UC	(0x00000004)	/* User Code */			    
#define MCF5307_RAMBAR_AS_UD	(0x00000002)	/* User Data */			    
#define MCF5307_RAMBAR_V	(0x00000001)		/* Valid */ 					

#define MCF5307_MBAR_BASE(a)	((a)&0xFFFFFC00)	/* Base Address */		    
#define MCF5307_MBAR_WP		(0x00000200)		/* Write Protect */				
#define MCF5307_MBAR_SC		(0x00000010)		/* Mask Supervisor Code Space */	
#define MCF5307_MBAR_SD		(0x00000008)		/* Mask Supervisor Data Space */	
#define MCF5307_MBAR_UC		(0x00000004)		/* Mask User Code Space */		    
#define MCF5307_MBAR_UD		(0x00000002)		/* Mask User Data Space	*/	    
#define MCF5307_MBAR_V		(0x00000001)		/* Valid */						

/**********************************************************************/
/*	System Integration Modules, SIM									  */
/**********************************************************************/

/*	Offsets of the registers from the MBAR */ 
#define MCF5307_SIM_RSR			(0x0000)
#define MCF5307_SIM_SYPCR		(0x0001)
#define MCF5307_SIM_SWIVR		(0x0002)
#define MCF5307_SIM_SWSR		(0x0003)
#define MCF5307_SIM_PAR			(0x0004)
#define MCF5307_SIM_IRQPAR		(0x0006)
#define MCF5307_SIM_PLLCR		(0x0008)
#define MCF5307_SIM_MPARK		(0x000C)
#define MCF5307_SIM_IPR			(0x0040)
#define MCF5307_SIM_IMR			(0x0044)
#define MCF5307_SIM_AVCR		(0x004B)
#define MCF5307_SIM_ICR0		(0x004C)
#define MCF5307_SIM_ICR1		(0x004D)
#define MCF5307_SIM_ICR2		(0x004E)
#define MCF5307_SIM_ICR3		(0x004F)
#define MCF5307_SIM_ICR4		(0x0050)
#define MCF5307_SIM_ICR5		(0x0051)
#define MCF5307_SIM_ICR6		(0x0052)
#define MCF5307_SIM_ICR7		(0x0053)
#define MCF5307_SIM_ICR8		(0x0054)
#define MCF5307_SIM_ICR9		(0x0055)
#define MCF5307_SIM_ICR10		(0x0056)
#define MCF5307_SIM_ICR11		(0x0057)

/* Read access macros for general use */
#define MCF5307_RD_SIM_RSR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_RSR,8)
#define MCF5307_RD_SIM_SYPCR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_SYPCR,8)
#define MCF5307_RD_SIM_SWIVR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_SWIVR,8)
#define MCF5307_RD_SIM_PAR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_PAR,16)
#define MCF5307_RD_SIM_IRQPAR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_IRQPAR,8)
#define MCF5307_RD_SIM_PLLCR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_PLLCR,8)
#define MCF5307_RD_SIM_MPARK(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_MPARK,8)
#define MCF5307_RD_SIM_IPR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_IPR,32)
#define MCF5307_RD_SIM_IMR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_IMR,32)
#define MCF5307_RD_SIM_AVCR(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_AVCR,8)
#define MCF5307_RD_SIM_ICR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR0,8)
#define MCF5307_RD_SIM_ICR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR1,8)
#define MCF5307_RD_SIM_ICR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR2,8)
#define MCF5307_RD_SIM_ICR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR3,8)
#define MCF5307_RD_SIM_ICR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR4,8)
#define MCF5307_RD_SIM_ICR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR5,8)
#define MCF5307_RD_SIM_ICR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR6,8)
#define MCF5307_RD_SIM_ICR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR7,8)
#define MCF5307_RD_SIM_ICR8(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR8,8)
#define MCF5307_RD_SIM_ICR9(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR9,8)
#define MCF5307_RD_SIM_ICR10(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR10,8)
#define MCF5307_RD_SIM_ICR11(IMMP)	Mcf5307_iord(IMMP,MCF5307_SIM_ICR11,8)

/* Write access macros for general use */ 
#define MCF5307_WR_SIM_RSR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_RSR,8,DATA)
#define MCF5307_WR_SIM_SYPCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_SYPCR,8,DATA)
#define MCF5307_WR_SIM_SWIVR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_SWIVR,8,DATA)
#define MCF5307_WR_SIM_SWSR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_SWSR,8,DATA)
#define MCF5307_WR_SIM_PAR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_PAR,16,DATA)
#define MCF5307_WR_SIM_IRQPAR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_IRQPAR,8,DATA)
#define MCF5307_WR_SIM_PLLCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_PLLCR,8,DATA)
#define MCF5307_WR_SIM_MPARK(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_MPARK,8,DATA)
#define MCF5307_WR_SIM_IPR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_IPR,32,DATA)
#define MCF5307_WR_SIM_IMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_IMR,32,DATA)
#define MCF5307_WR_SIM_AVCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_AVCR,8,DATA)
#define MCF5307_WR_SIM_ICR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR0,8,DATA)
#define MCF5307_WR_SIM_ICR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR1,8,DATA)
#define MCF5307_WR_SIM_ICR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR2,8,DATA)
#define MCF5307_WR_SIM_ICR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR3,8,DATA)
#define MCF5307_WR_SIM_ICR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR4,8,DATA)
#define MCF5307_WR_SIM_ICR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR5,8,DATA)
#define MCF5307_WR_SIM_ICR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR6,8,DATA)
#define MCF5307_WR_SIM_ICR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR7,8,DATA)
#define MCF5307_WR_SIM_ICR8(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR8,8,DATA)
#define MCF5307_WR_SIM_ICR9(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR9,8,DATA)
#define MCF5307_WR_SIM_ICR10(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR10,8,DATA)
#define MCF5307_WR_SIM_ICR11(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_SIM_ICR11,8,DATA)

typedef volatile struct
{
	uint8	RSR;		/* Reset Status Register */		    
	uint8	SYPCR;		/* System Protection Control Register */	    
	uint8	SWIVR;		/* Software Watchdog Interrupt Vector Reg */   
	uint8	SWSR;		/* Software Watchdog Service Register */	    
	uint16	PAR;		/* Pin Assignment Register */		    
	uint8	IRQPAR;		/* Interrupt Assignment Register */	    
	uint8	reserved0;
	uint8	PLLCR;		/* Phase Locked Loop Control Register */	    
	uint8	reserved1;
	uint16	reserved2;
	uint8	MARBCR;		/* MBus Park Register */			    
	uint8	reserved3;
	uint16	reserved4;
	uint32	reserved5[12];
	uint32	IPR;		/* Interrupt Pending Register */		    
	uint32	IMR;		/* Internal Mask Register */		    
	uint8	reserved6[3];
	uint8	AVCR;		/* Auto Vector Control Register */		    
	uint8	ICR0;		/* Interrupt Control Register:  SWT */	    
	uint8	ICR1;		/* Interrupt Control Register:  Timer 1 */	    
	uint8	ICR2;		/* Interrupt Control Register:  Timer 2 */	    
	uint8	ICR3;		/* Interrupt Control Register:  MBus */	    
	uint8	ICR4;		/* Interrupt Control Register:  UART 1 */	    
	uint8	ICR5;		/* Interrupt Control Register:  UART 2 */	    
	uint8	ICR6;		/* Interrupt Control Register:  DMA 0 */	    
	uint8	ICR7;		/* Interrupt Control Register:  DMA 1 */	    
	uint8	ICR8;		/* Interrupt Control Register:  DMA 2 */	    
	uint8	ICR9;		/* Interrupt Control Register:  DMA 3 */	    
	uint8	ICR10;		/* Interrupt Control Register:  Reserved */    
	uint8	ICR11;		/* Interrupt Control Register:  Reserved */    
} MCF5307_SIM;

#define MCF5307_SIM_RSR_HRST		(0x80)	/* Hard or System Reset */		    
#define MCF5307_SIM_RSR_SWTR		(0x20)	/* Software Watchdog Timer Reset */    

#define MCF5307_SIM_SYPCR_SWE		(0x80)	/* Software Watchdog Enable */	    
#define MCF5307_SIM_SYPCR_SWRI		(0x40)	/* SW Reset/Interrupt Select */	    
#define MCF5307_SIM_SYPCR_SWT_2_9	(0x00)	/* SW Timeout: 2^9  / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_11	(0x08)	/* SW Timeout: 2^11 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_13	(0x10)	/* SW Timeout: 2^13 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_15	(0x18)	/* SW Timeout: 2^15 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_18	(0x20)	/* SW Timeout: 2^18 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_20	(0x28)	/* SW Timeout: 2^20 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_22	(0x30)	/* SW Timeout: 2^22 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWT_2_24	(0x38)	/* SW Timeout: 2^24 / Sys Freq */	    
#define MCF5307_SIM_SYPCR_SWTA		(0x04)	/* SW Transfer Acknowledge Enable */   
#define MCF5307_SIM_SYPCR_SWTAVAL	(0x02)	/* SW Transfer Acknowledge Valid */    

#define MCF5307_SIM_SWSR_55			(0x55)		/* Write $55 to SWSR */	    
#define MCF5307_SIM_SWSR_AA			(0xaa)		/* Write $AA to SWSR */	    
	
#define MCF5307_SIM_PAR_ADDR31		(0x8000)	/* Assign Pin as ADDR31 */	    
#define MCF5307_SIM_PAR_ADDR30		(0x4000)	/* Assign Pin as ADDR30 */	    
#define MCF5307_SIM_PAR_ADDR29		(0x2000)	/* Assign Pin as ADDR29 */	    
#define MCF5307_SIM_PAR_ADDR28		(0x1000)	/* Assign Pin as ADDR28 */	    
#define MCF5307_SIM_PAR_ADDR27		(0x0800)	/* Assign Pin as ADDR27 */	    
#define MCF5307_SIM_PAR_ADDR26		(0x0400)	/* Assign Pin as ADDR26 */	    
#define MCF5307_SIM_PAR_ADDR25		(0x0200)	/* Assign Pin as ADDR25 */	    
#define MCF5307_SIM_PAR_ADDR24		(0x0100)	/* Assign Pin as ADDR24 */	    
#define MCF5307_SIM_PAR_XTIP		(0x0080)	/* Assign Pin as XTIP */	    
#define MCF5307_SIM_PAR_DREQ0		(0x0040)	/* Assign Pin as DREQ0 */	    
#define MCF5307_SIM_PAR_DREQ1		(0x0020)	/* Assign Pin as DREQ1 */	    
#define MCF5307_SIM_PAR_TM2			(0x0010)	/* Assign Pin as TM2 */	    
#define MCF5307_SIM_PAR_TM1			(0x0008)	/* Assign Pin as TM1 */	    
#define MCF5307_SIM_PAR_TM0			(0x0004)	/* Assign Pin as TM0 */	    
#define MCF5307_SIM_PAR_TT1			(0x0002)	/* Assign Pin as TT1 */	    
#define MCF5307_SIM_PAR_TT0			(0x0001)	/* Assign Pin as TT0 */	    
	
#define MCF5307_SIM_IRQPAR_2		(0x80)		/* IRQ[5] pin to Int IL 4, not 5 */    
#define MCF5307_SIM_IRQPAR_1		(0x20)		/* IRQ[3] pin to Int IL 6, not 3 */    
#define MCF5307_SIM_IRQPAR_0		(0x10)		/* IRQ[1] pin to Int IL 2, not 1 */    

#define MCF5307_SIM_PLLCR_ENBSTOP	(0x80)			/* Enable CPU STOP Instruction */	    
#define MCF5307_SIM_PLLCR_PLLIPL(a)	(((a)&0x07)<<4)	/* PLL Wake-up IPL */	    

#define MCF5307_SIM_MPARK_PARK(a)	(((a)&0x03)<<6)	/* Default Bus Master */	    
#define MCF5307_SIM_MPARK_E2MCTRL	(0x20)			/* EBus to MBus Arbitration */ 
#define MCF5307_SIM_MPARK_EARBCTRL	(0x10)			/* SBus to EBus Arbitration */ 
#define MCF5307_SIM_MPARK_SHOWDATA	(0x08)			/* Show SBus on EBus */	
#define MCF5307_SIM_MPARK_BCR24BIT	(0x01)			/* BCR is 24 or 16 bit register */    

#define MCF5307_SIM_IPR_DMA3		(0x00020000)	/* Interrupt Pending DMA3 */   
#define MCF5307_SIM_IPR_DMA2		(0x00010000)	/* Interrupt Pending DMA2 */   
#define MCF5307_SIM_IPR_DMA1		(0x00008000)	/* Interrupt Pending DMA1 */  
#define MCF5307_SIM_IPR_DMA0		(0x00004000)	/* Interrupt Pending DMA0 */   
#define MCF5307_SIM_IPR_UART2		(0x00002000)	/* Interrupt Pending UART2 */  
#define MCF5307_SIM_IPR_UART1		(0x00001000)	/* Interrupt Pending UART1 */  
#define MCF5307_SIM_IPR_MBUS		(0x00000800)	/* Interrupt Pending MBUS */   
#define MCF5307_SIM_IPR_TIMER2		(0x00000400)	/* Interrupt Pending TIMER2 */ 
#define MCF5307_SIM_IPR_TIMER1		(0x00000200)	/* Interrupt Pending TIMER1 */ 
#define MCF5307_SIM_IPR_SWT			(0x00000100)	/* Interrupt Pending SWT */    
#define MCF5307_SIM_IPR_EINT7		(0x00000080)	/* Interrupt Pending EINT7 */  
#define MCF5307_SIM_IPR_EINT6		(0x00000040)	/* Interrupt Pending EINT6 */  
#define MCF5307_SIM_IPR_EINT5		(0x00000020)	/* Interrupt Pending EINT5 */  
#define MCF5307_SIM_IPR_EINT4		(0x00000010)	/* Interrupt Pending EINT4 */  
#define MCF5307_SIM_IPR_EINT3		(0x00000008)	/* Interrupt Pending EINT3 */  
#define MCF5307_SIM_IPR_EINT2		(0x00000004)	/* Interrupt Pending EINT2 */ 
#define MCF5307_SIM_IPR_EINT1		(0x00000002)	/* Interrupt Pending EINT1 */ 

#define MCF5307_SIM_IMR_DMA3		(0x00020000)	/* Mask DMA3 */		    
#define MCF5307_SIM_IMR_DMA2		(0x00010000)	/* Mask DMA2 */		    
#define MCF5307_SIM_IMR_DMA1		(0x00008000)	/* Mask DMA1 */		    
#define MCF5307_SIM_IMR_DMA0		(0x00004000)	/* Mask DMA0 */		    
#define MCF5307_SIM_IMR_UART2		(0x00002000)	/* Mask UART2 */		    
#define MCF5307_SIM_IMR_UART1		(0x00001000)	/* Mask UART1 */		    
#define MCF5307_SIM_IMR_MBUS		(0x00000800)	/* Mask MBUS */		    
#define MCF5307_SIM_IMR_TIMER2		(0x00000400)	/* Mask TIMER2 */	    
#define MCF5307_SIM_IMR_TIMER1		(0x00000200)	/* Mask TIMER1 */		    
#define MCF5307_SIM_IMR_SWT			(0x00000100)	/* Mask SWT */		    
#define MCF5307_SIM_IMR_EINT7		(0x00000080)	/* Mask EINT7 */		    
#define MCF5307_SIM_IMR_EINT6		(0x00000040)	/* Mask EINT6 */		    
#define MCF5307_SIM_IMR_EINT5		(0x00000020)	/* Mask EINT5 */		    
#define MCF5307_SIM_IMR_EINT4		(0x00000010)	/* Mask EINT4 */		    
#define MCF5307_SIM_IMR_EINT3		(0x00000008)	/* Mask EINT3 */		    
#define MCF5307_SIM_IMR_EINT2		(0x00000004)	/* Mask EINT2 */		    
#define MCF5307_SIM_IMR_EINT1		(0x00000002)	/* Mask EINT1 */		    

#define MCF5307_SIM_AVCR_AVEC7		(0x80)			/* Auto Vector Ext Interrupt 7 */	    
#define MCF5307_SIM_AVCR_AVEC6		(0x40)			/* Auto Vector Ext Interrupt 6 */	    
#define MCF5307_SIM_AVCR_AVEC5		(0x20)			/* Auto Vector Ext Interrupt 5 */	    
#define MCF5307_SIM_AVCR_AVEC4		(0x10)			/* Auto Vector Ext Interrupt 4 */	    
#define MCF5307_SIM_AVCR_AVEC3		(0x08)			/* Auto Vector Ext Interrupt 3 */	    
#define MCF5307_SIM_AVCR_AVEC2		(0x04)			/* Auto Vector Ext Interrupt 2 */	    
#define MCF5307_SIM_AVCR_AVEC1		(0x02)			/* Auto Vector Ext Interrupt 1 */	    
#define MCF5307_SIM_AVCR_BLK		(0x01)			/* Block Address Strobe */		    

#define MCF5307_SIM_ICR_AVEC		(0x80)			/* Autovector Enable */	    
#define MCF5307_SIM_ICR_IL(a)		(((a)&0x07)<<2)	/* Interrupt Level */	    
#define MCF5307_SIM_ICR_IP_EXT		(0x02)			/* High Priority External */  
#define MCF5307_SIM_ICR_IP_INT		(0x01)			/* High Priority Internal */   

/**********************************************************************/
/*	Chip-Select Module, CS											  */
/**********************************************************************/

#if (defined(MCF5307A))

/* Offsets of the registers from the MBAR */
#define MCF5307_CS_CSAR0		(0x0080)
#define MCF5307_CS_CSMR0		(0x0084)
#define MCF5307_CS_CSCR0		(0x008A)

#define MCF5307_CS_CSAR1		(0x008C)
#define MCF5307_CS_CSMR1		(0x0090)
#define MCF5307_CS_CSCR1		(0x0096)

#define MCF5307_CS_CSAR2		(0x0098)
#define MCF5307_CS_CSMR2		(0x009C)
#define MCF5307_CS_CSCR2		(0x00A2)

#define MCF5307_CS_CSAR3		(0x00A4)
#define MCF5307_CS_CSMR3		(0x00A8)
#define MCF5307_CS_CSCR3		(0x00AE)

#define MCF5307_CS_CSAR4		(0x00B0)
#define MCF5307_CS_CSMR4		(0x00B4)
#define MCF5307_CS_CSCR4		(0x00BA)

#define MCF5307_CS_CSAR5		(0x00BC)
#define MCF5307_CS_CSMR5		(0x00C0)
#define MCF5307_CS_CSCR5		(0x00C6)

#define MCF5307_CS_CSAR6		(0x00C8)
#define MCF5307_CS_CSMR6		(0x00CC)
#define MCF5307_CS_CSCR6		(0x00D2)

#define MCF5307_CS_CSAR7		(0x00D4)
#define MCF5307_CS_CSMR7		(0x00D8)
#define MCF5307_CS_CSCR7		(0x00DE)

/* Read access macros for general use */ 
#define MCF5307_RD_CS_CSAR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR0,16)
#define MCF5307_RD_CS_CSMR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR0,32)
#define MCF5307_RD_CS_CSCR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR0,16)

#define MCF5307_RD_CS_CSAR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR1,16)
#define MCF5307_RD_CS_CSMR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR1,32)
#define MCF5307_RD_CS_CSCR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR1,16)

#define MCF5307_RD_CS_CSAR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR2,16)
#define MCF5307_RD_CS_CSMR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR2,32)
#define MCF5307_RD_CS_CSCR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR2,16)

#define MCF5307_RD_CS_CSAR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR3,16)
#define MCF5307_RD_CS_CSMR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR3,32)
#define MCF5307_RD_CS_CSCR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR3,16)

#define MCF5307_RD_CS_CSAR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR4,16)
#define MCF5307_RD_CS_CSMR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR4,32)
#define MCF5307_RD_CS_CSCR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR4,16)

#define MCF5307_RD_CS_CSAR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR5,16)
#define MCF5307_RD_CS_CSMR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR5,32)
#define MCF5307_RD_CS_CSCR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR5,16)

#define MCF5307_RD_CS_CSAR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR6,16)
#define MCF5307_RD_CS_CSMR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR6,32)
#define MCF5307_RD_CS_CSCR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR6,16)

#define MCF5307_RD_CS_CSAR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR7,16)
#define MCF5307_RD_CS_CSMR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR7,32)
#define MCF5307_RD_CS_CSCR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR7,16)

/* Write access macros for general use */
#define MCF5307_WR_CS_CSAR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR0,16,DATA)
#define MCF5307_WR_CS_CSMR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR0,32,DATA)
#define MCF5307_WR_CS_CSCR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR0,16,DATA)

#define MCF5307_WR_CS_CSAR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR1,16,DATA)
#define MCF5307_WR_CS_CSMR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR1,32,DATA)
#define MCF5307_WR_CS_CSCR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR1,16,DATA)

#define MCF5307_WR_CS_CSAR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR2,16,DATA)
#define MCF5307_WR_CS_CSMR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR2,32,DATA)
#define MCF5307_WR_CS_CSCR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR2,16,DATA)

#define MCF5307_WR_CS_CSAR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR3,16,DATA)
#define MCF5307_WR_CS_CSMR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR3,32,DATA)
#define MCF5307_WR_CS_CSCR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR3,16,DATA)

#define MCF5307_WR_CS_CSAR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR4,16,DATA)
#define MCF5307_WR_CS_CSMR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR4,32,DATA)
#define MCF5307_WR_CS_CSCR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR4,16,DATA)

#define MCF5307_WR_CS_CSAR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR5,16,DATA)
#define MCF5307_WR_CS_CSMR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR5,32,DATA)
#define MCF5307_WR_CS_CSCR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR5,16,DATA)

#define MCF5307_WR_CS_CSAR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR6,16,DATA)
#define MCF5307_WR_CS_CSMR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR6,32,DATA)
#define MCF5307_WR_CS_CSCR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR6,16,DATA)

#define MCF5307_WR_CS_CSAR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR7,16,DATA)
#define MCF5307_WR_CS_CSMR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR7,32,DATA)
#define MCF5307_WR_CS_CSCR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR7,16,DATA)

#else	/* H55J mask */

/* Offsets of the registers from the MBAR */
#define MCF5307_CS_CSAR0		(0x0080)
#define MCF5307_CS_CSMR0		(0x0084)
#define MCF5307_CS_CSCR0		(0x008A)
#define MCF5307_CS_CSAR1		(0x008C)
#define MCF5307_CS_CSMR1		(0x0090)
#define MCF5307_CS_CSCR1		(0x0096)
#define MCF5307_CS_CSBAR		(0x0098)
#define MCF5307_CS_CSBMR		(0x009C)
#define MCF5307_CS_CSMR2		(0x009E)
#define MCF5307_CS_CSCR2		(0x00A2)
#define MCF5307_CS_CSMR3		(0x00AA)
#define MCF5307_CS_CSCR3		(0x00AE)
#define MCF5307_CS_CSMR4		(0x00B6)
#define MCF5307_CS_CSCR4		(0x00BA)
#define MCF5307_CS_CSMR5		(0x00C2)
#define MCF5307_CS_CSCR5		(0x00C6)
#define MCF5307_CS_CSMR6		(0x00CE)
#define MCF5307_CS_CSCR6		(0x00D2)
#define MCF5307_CS_CSMR7		(0x00DA)
#define MCF5307_CS_CSCR7		(0x00DE)

/* Read access macros for general use */ 
#define MCF5307_RD_CS_CSAR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR0,16)
#define MCF5307_RD_CS_CSMR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR0,32)
#define MCF5307_RD_CS_CSCR0(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR0,16)
#define MCF5307_RD_CS_CSAR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSAR1,16)
#define MCF5307_RD_CS_CSMR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR1,32)
#define MCF5307_RD_CS_CSCR1(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR1,16)
#define MCF5307_RD_CS_CSBAR(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSBAR,8)
#define MCF5307_RD_CS_CSBMR(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSBMR,8)
#define MCF5307_RD_CS_CSMR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR2,16)
#define MCF5307_RD_CS_CSCR2(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR2,16)
#define MCF5307_RD_CS_CSMR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR3,16)
#define MCF5307_RD_CS_CSCR3(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR3,16)
#define MCF5307_RD_CS_CSMR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR4,16)
#define MCF5307_RD_CS_CSCR4(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR4,16)
#define MCF5307_RD_CS_CSMR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR5,16)
#define MCF5307_RD_CS_CSCR5(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR5,16)
#define MCF5307_RD_CS_CSMR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR6,16)
#define MCF5307_RD_CS_CSCR6(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR6,16)
#define MCF5307_RD_CS_CSMR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSMR7,16)
#define MCF5307_RD_CS_CSCR7(IMMP)	Mcf5307_iord(IMMP,MCF5307_CS_CSCR7,16)

/* Write access macros for general use */
#define MCF5307_WR_CS_CSAR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR0,16,DATA)
#define MCF5307_WR_CS_CSMR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR0,32,DATA)
#define MCF5307_WR_CS_CSCR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR0,16,DATA)
#define MCF5307_WR_CS_CSAR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSAR1,16,DATA)
#define MCF5307_WR_CS_CSMR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR1,32,DATA)
#define MCF5307_WR_CS_CSCR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR1,16,DATA)
#define MCF5307_WR_CS_CSBAR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSBAR,8,DATA)
#define MCF5307_WR_CS_CSBMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSBMR,8,DATA)
#define MCF5307_WR_CS_CSMR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR2,16,DATA)
#define MCF5307_WR_CS_CSCR2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR2,16,DATA)
#define MCF5307_WR_CS_CSMR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR3,16,DATA)
#define MCF5307_WR_CS_CSCR3(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR3,16,DATA)
#define MCF5307_WR_CS_CSMR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR4,16,DATA)
#define MCF5307_WR_CS_CSCR4(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR4,16,DATA)
#define MCF5307_WR_CS_CSMR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR5,16,DATA)
#define MCF5307_WR_CS_CSCR5(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR5,16,DATA)
#define MCF5307_WR_CS_CSMR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR6,16,DATA)
#define MCF5307_WR_CS_CSCR6(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR6,16,DATA)
#define MCF5307_WR_CS_CSMR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSMR7,16,DATA)
#define MCF5307_WR_CS_CSCR7(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_CS_CSCR7,16,DATA)

#endif

typedef volatile struct
{
	uint32	reserved1[0x20];
	uint16	CSAR0;		/* Chip-Select Address Register	- Bank 0 */    
	uint16	reserved2;
	uint32	CSMR0;		/* Chip-Select Mask Register	- Bank 0 */    
	uint16	reserved3;
	uint16	CSCR0;		/* Chip-Select Control Register	- Bank 0 */    
	uint16	CSAR1;		/* Chip-Select Address Register	- Bank 1 */    
	uint16	reserved4;
	uint32	CSMR1;		/* Chip-Select Mask Register	- Bank 1 */    
	uint16	reserved5;
	uint16	CSCR1;		/* Chip-Select Control Register	- Bank 1 */    
	uint8	CSBAR;		/* Chip-Select Address Register	- Banks 2-7 */
	uint8	reserved6;
	uint16	reserved7;
	uint8	CSBMR;		/* Chip-Select Mask Register	- Banks 2-7 */
	uint8	reserveME;
	uint16	CSMR2;		/* Chip-Select Mask Register	- Bank 2 */    
	uint16	reserved8;
	uint16	CSCR2;		/* Chip-Select Control Register	- Bank 2 */    
	uint32	reserved9;
	uint16	reserved7b;
	uint16	CSMR3;		/* Chip-Select Mask Register	- Bank 3 */    
	uint16	reserved10;
	uint16	CSCR3;		/* Chip-Select Control Register	- Bank 3 */    
	uint32	reserved11;
	uint16	reserved10a;
	uint16	CSMR4;		/* Chip-Select Mask Register	- Bank 4 */    
	uint16	reserved12;
	uint16	CSCR4;		/* Chip-Select Control Register	- Bank 4 */    
	uint32	reserved13;
	uint16	reserved12a;
	uint16	CSMR5;		/* Chip-Select Mask Register	- Bank 5 */    
	uint16	reserved14;
	uint16	CSCR5;		/* Chip-Select Control Register	- Bank 5 */    
	uint32	reserved15;
	uint16	reserved16a;
	uint16	CSMR6;		/* Chip-Select Mask Register	- Bank 6 */    
	uint16	reserved16;
	uint16	CSCR6;		/* Chip-Select Control Register	- Bank 6 */    
	uint32	reserved17;
	uint16	reserved18a;
	uint16	CSMR7;		/* Chip-Select Mask Register	- Bank 7 */    
	uint16	reserved18;
	uint16	CSCR7;		/* Chip-Select Control Register	- Bank 7 */    
} MCF5307_CS;

/*  These definitions only exists in the CSMR for Banks 0 and 1. */		
#define MCF5307_CS_CSMR_MASK_4G		(0xFFFF0000)	/* Set Bank to 4G */	
#define MCF5307_CS_CSMR_MASK_2G		(0x7FFF0000)	/* Set Bank to 2G */	
#define MCF5307_CS_CSMR_MASK_1G		(0x3FFF0000)	/* Set Bank to 1G */	
#define MCF5307_CS_CSMR_MASK_1024M	(0x3FFF0000)	/* Set Bank to 1024M */
#define MCF5307_CS_CSMR_MASK_512M	(0x1FFF0000)	/* Set Bank to 512M */	
#define MCF5307_CS_CSMR_MASK_256M	(0x0FFF0000)	/* Set Bank to 256M */	
#define MCF5307_CS_CSMR_MASK_128M	(0x07FF0000)	/* Set Bank to 128M */	
#define MCF5307_CS_CSMR_MASK_64M	(0x03FF0000)	/* Set Bank to 64M */	
#define MCF5307_CS_CSMR_MASK_32M	(0x01FF0000)	/* Set Bank to 32M */	
#define MCF5307_CS_CSMR_MASK_16M	(0x00FF0000)	/* Set Bank to 16M */	
#define MCF5307_CS_CSMR_MASK_8M		(0x007F0000)	/* Set Bank to 8M */	
#define MCF5307_CS_CSMR_MASK_4M		(0x003F0000)	/* Set Bank to 4M */	
#define MCF5307_CS_CSMR_MASK_2M		(0x001F0000)	/* Set Bank to 2M */	
#define MCF5307_CS_CSMR_MASK_1M		(0x000F0000)	/* Set Bank to 1M */	
#define MCF5307_CS_CSMR_MASK_1024K	(0x000F0000)	/* Set Bank to 1024K */
#define MCF5307_CS_CSMR_MASK_512K	(0x00070000)	/* Set Bank to 512K	*/
#define MCF5307_CS_CSMR_MASK_256K	(0x00030000)	/* Set Bank to 256K */
#define MCF5307_CS_CSMR_MASK_128K	(0x00010000)	/* Set Bank to 128K */
#define MCF5307_CS_CSMR_MASK_64K	(0x00000000)	/* Set Bank to 64K */	
#define MCF5307_CS_CSMR_CPU			(0x00000020)	/* CPU and IACK Cycle Mask */ 

/*  The following definitions exist for all Banks 0-7 */					
#define MCF5307_CS_CSAR(a)		(((a)&0xFFFF0000)>>16)	/* Base Address */		
#define MCF5307_CS_CSBARx(a)	(((a)&0xFF000000)>>24)	/* Base for CS2-7 */	

#define MCF5307_CS_CSMR_WP		(0x00000100)	/* Write Protect */			
#define MCF5307_CS_CSMR_AM		(0x00000040)	/* Alternate Master Mask */	
#define MCF5307_CS_CSMR_SC		(0x00000010)	/* Supervisor Code Mask */		
#define MCF5307_CS_CSMR_SD		(0x00000008)	/* Supervisor Data Mask */		
#define MCF5307_CS_CSMR_UC		(0x00000004)	/* User Code Mask */			
#define MCF5307_CS_CSMR_UD		(0x00000002)	/* User Data Mask */			
#define MCF5307_CS_CSMR_V		(0x00000001)	/* Valid Register */			

#define MCF5307_CS_CSCR_WS(a)	(((a)&0x0F)<<10) /* Wait States */				
#define MCF5307_CS_CSCR_AA		(0x0100)		/* Auto Acknowledge Enable */	
#define MCF5307_CS_CSCR_PS_8	(0x0040)		/* Port Size:   8-bit */		
#define MCF5307_CS_CSCR_PS_16	(0x0080)		/* Port Size:  16-bit */		
#define MCF5307_CS_CSCR_PS_32	(0x0000)		/* Port Size:  32-bit */		
#define MCF5307_CS_CSCR_BEM		(0x0020)		/* Byte Module Enable */		
#define MCF5307_CS_CSCR_BSTR	(0x0010)		/* Burst Read Enable */		
#define MCF5307_CS_CSCR_BSTW	(0x0008)		/* Burst Write Enable */		

/**********************************************************************/
/*	Parallel Port (General Purpose I/O) Module, PP					  */
/**********************************************************************/

/* Offsets of the registers from the MBAR */ 
#define MCF5307_PP_PADDR		(0x0244)
#define MCF5307_PP_PADAT		(0x0248)

/* Read access macros for general use */ 
#define MCF5307_RD_PP_PADDR(IMMP)	Mcf5307_iord(IMMP,MCF5307_PP_PADDR,16)
#define MCF5307_RD_PP_PADAT(IMMP)	Mcf5307_iord(IMMP,MCF5307_PP_PADAT,16)

/* Write access macros for general use */
#define MCF5307_WR_PP_PADDR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_PP_PADDR,16,DATA)
#define MCF5307_WR_PP_PADAT(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_PP_PADAT,16,DATA)

typedef volatile struct
{
	uint32	reserved0[0x91];
	uint16	PADDR;		/* Port A Data Direction Register */	
	uint16	reserved1;
	uint16	PADAT;		/* Port A Data Register */			
	uint16	reserved2;
} MCF5307_PP;

#define MCF5307_PP_PADDR_15	(0x8000)	/* Bit 15 General I/O Output */	
#define MCF5307_PP_PADDR_14	(0x4000)	/* Bit 14 General I/O Output */	
#define MCF5307_PP_PADDR_13	(0x2000)	/* Bit 13 General I/O Output */	
#define MCF5307_PP_PADDR_12	(0x1000)	/* Bit 12 General I/O Output */	
#define MCF5307_PP_PADDR_11	(0x0800)	/* Bit 11 General I/O Output */	
#define MCF5307_PP_PADDR_10	(0x0400)	/* Bit 10 General I/O Output */	
#define MCF5307_PP_PADDR_9	(0x0200)	/* Bit  9 General I/O Output */	
#define MCF5307_PP_PADDR_8	(0x0100)	/* Bit  8 General I/O Output */	
#define MCF5307_PP_PADDR_7	(0x0080)	/* Bit  7 General I/O Output */	
#define MCF5307_PP_PADDR_6	(0x0040)	/* Bit  6 General I/O Output */	
#define MCF5307_PP_PADDR_5	(0x0020)	/* Bit  5 General I/O Output */	
#define MCF5307_PP_PADDR_4	(0x0010)	/* Bit  4 General I/O Output */	
#define MCF5307_PP_PADDR_3	(0x0008)	/* Bit  3 General I/O Output */	
#define MCF5307_PP_PADDR_2	(0x0004)	/* Bit  2 General I/O Output */	
#define MCF5307_PP_PADDR_1	(0x0002)	/* Bit  1 General I/O Output */	
#define MCF5307_PP_PADDR_0	(0x0001)	/* Bit  0 General I/O Output */	

#define MCF5307_PP_PADAT_15	(0x8000)	/* Bit 15 Current Status */
#define MCF5307_PP_PADAT_14	(0x4000)	/* Bit 14 Current Status */	
#define MCF5307_PP_PADAT_13	(0x2000)	/* Bit 13 Current Status */	
#define MCF5307_PP_PADAT_12	(0x1000)	/* Bit 12 Current Status */	
#define MCF5307_PP_PADAT_11	(0x0800)	/* Bit 11 Current Status */	
#define MCF5307_PP_PADAT_10	(0x0400)	/* Bit 10 Current Status */	
#define MCF5307_PP_PADAT_9	(0x0200)	/* Bit  9 Current Status */	
#define MCF5307_PP_PADAT_8	(0x0100)	/* Bit  8 Current Status */	
#define MCF5307_PP_PADAT_7	(0x0080)	/* Bit  7 Current Status */	
#define MCF5307_PP_PADAT_6	(0x0040)	/* Bit  6 Current Status */	
#define MCF5307_PP_PADAT_5	(0x0020)	/* Bit  5 Current Status */	
#define MCF5307_PP_PADAT_4	(0x0010)	/* Bit  4 Current Status */	
#define MCF5307_PP_PADAT_3	(0x0008)	/* Bit  3 Current Status */	
#define MCF5307_PP_PADAT_2	(0x0004)	/* Bit  2 Current Status */	
#define MCF5307_PP_PADAT_1	(0x0002)	/* Bit  1 Current Status */	
#define MCF5307_PP_PADAT_0	(0x0001)	/* Bit  0 Current Status */	

/**********************************************************************/
/*	DRAM Controller Module, DRAMC									  */
/**********************************************************************/

/* Offsets of the registers from the MBAR */
#define MCF5307_DRAMC_DCR	(0x0100)
#define MCF5307_DRAMC_DACR0	(0x0108)
#define MCF5307_DRAMC_DMR0	(0x010C)
#define MCF5307_DRAMC_DACR1	(0x0110)
#define MCF5307_DRAMC_DMR1	(0x0114)

/* Read access macros for general use */
#define MCF5307_RD_DRAMC_DCR(IMMP)		\
	Mcf5307_iord(IMMP,MCF5307_DRAMC_DCR,16)
#define MCF5307_RD_DRAMC_DACR0(IMMP)	\
	Mcf5307_iord(IMMP,MCF5307_DRAMC_DACR0,32)
#define MCF5307_RD_DRAMC_DMR0(IMMP)		\
	Mcf5307_iord(IMMP,MCF5307_DRAMC_DMR0,32)
#define MCF5307_RD_DRAMC_DACR1(IMMP)	\
	Mcf5307_iord(IMMP,MCF5307_DRAMC_DACR1,32)
#define MCF5307_RD_DRAMC_DMR1(IMMP)		\
	Mcf5307_iord(IMMP,MCF5307_DRAMC_DMR1,32)

/* Write access macros for general use */
#define MCF5307_WR_DRAMC_DCR(IMMP,DATA)		\
	Mcf5307_iowr(IMMP,MCF5307_DRAMC_DCR,16,DATA)
#define MCF5307_WR_DRAMC_DACR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_DRAMC_DACR0,32,DATA)
#define MCF5307_WR_DRAMC_DMR0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_DRAMC_DMR0,32,DATA)
#define MCF5307_WR_DRAMC_DACR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_DRAMC_DACR1,32,DATA)
#define MCF5307_WR_DRAMC_DMR1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_DRAMC_DMR1,32,DATA)

typedef volatile struct
{
	uint32	reserved1[0x40];
	uint16	DCR;		/* DRAM Control Register */				
	uint16	reserved2;
	uint32	reserved3;
	uint32	DACR0;		/* DRAM Address and Control Register 0 */	
	uint32	DMR0;		/* DRAM Controller Mask Register 0 */		
	uint32	DACR1;		/* DRAM Address and Control Register 1 */	
	uint32	DMR1;		/* DRAM Controller Mask Register 1 */		
} MCF5307_DRAMC;
 
/* Controls used by both Synchronous and Asynchronous DRAM */			
#define MCF5307_DRAMC_DCR_SO			(0x8000)		/* Synchronous Operation */	
#define MCF5307_DRAMC_DCR_NAM			(0x2000)		/* No Address Multiplexing */	
#define MCF5307_DRAMC_DCR_RC(a)			((a)&0x01FF)	/* Refresh Count */			

#define MCF5307_DRAMC_DACR_BASE(a)		((a)&0xFFFC0000) /* Base Address */			
#define MCF5307_DRAMC_DACR_RE			(0x00008000)	/* Refresh Enable */			
#define MCF5307_DRAMC_DACR_PS_32		(0x00000000)	/* Port Size:  32-bit */		
#define MCF5307_DRAMC_DACR_PS_8			(0x00000010)	/* Port Size:   8-bit */		
#define MCF5307_DRAMC_DACR_PS_16		(0x00000020)	/* Port Size:  16-bit */		

#define MCF5307_DRAMC_DCMR_MASK_4G		(0xFFFC0000)	/* DRAM Size of 4G */			
#define MCF5307_DRAMC_DCMR_MASK_2G		(0x7FFC0000)	/* DRAM Size of 2G */			
#define MCF5307_DRAMC_DCMR_MASK_1G		(0x3FFC0000)	/* DRAM Size of 1G */			
#define MCF5307_DRAMC_DCMR_MASK_1024M	(0x3FFC0000)	/* DRAM Size of 1024M */	
#define MCF5307_DRAMC_DCMR_MASK_512M	(0x1FFC0000)	/* DRAM Size of 512M */	
#define MCF5307_DRAMC_DCMR_MASK_256M	(0x0FFC0000)	/* DRAM Size of 256M */	
#define MCF5307_DRAMC_DCMR_MASK_128M	(0x07FC0000)	/* DRAM Size of 128M */	
#define MCF5307_DRAMC_DCMR_MASK_64M		(0x03FC0000)	/* DRAM Size of 64M */			
#define MCF5307_DRAMC_DCMR_MASK_32M		(0x01FC0000)	/* DRAM Size of 32M */			
#define MCF5307_DRAMC_DCMR_MASK_16M		(0x00FC0000)	/* DRAM Size of 16M	*/		
#define MCF5307_DRAMC_DCMR_MASK_8M		(0x007C0000)	/* DRAM Size of 8M */			
#define MCF5307_DRAMC_DCMR_MASK_4M		(0x003C0000)	/* DRAM Size of 4M */			
#define MCF5307_DRAMC_DCMR_MASK_2M		(0x001C0000)	/* DRAM Size of 2M */			
#define MCF5307_DRAMC_DCMR_MASK_1M		(0x000C0000)	/* DRAM Size of 1M */			
#define MCF5307_DRAMC_DCMR_MASK_1024K	(0x00040000)	/* DRAM Size of 1024K */		
#define MCF5307_DRAMC_DCMR_MASK_512K	(0x00000000)	/* DRAM Size of 512K */		
#define MCF5307_DRAMC_DCMR_WP			(0x00000100)	/* Write Protect */		
#define MCF5307_DRAMC_DCMR_CPU			(0x00000040)	/* CPU Space Ignored */		
#define MCF5307_DRAMC_DCMR_AM			(0x00000020)	/* Alternate Master Ignored */	
#define MCF5307_DRAMC_DCMR_SC			(0x00000010)	/* Supervisor Code Ignored */	
#define MCF5307_DRAMC_DCMR_SD			(0x00000008)	/* Supervisor Data Ignored */	
#define MCF5307_DRAMC_DCMR_UC			(0x00000004)	/* User Code Ignored */		
#define MCF5307_DRAMC_DCMR_UD			(0x00000002)	/* User Data Ignored */		
#define MCF5307_DRAMC_DCMR_V			(0x00000001)	/* Valid Register */		

/* Controls used only by Asynchronous DRAM*/				
#define MCF5307_DRAMC_DCR_RRA_2			(0x0000)		/* Refresh RAS Asserted 2 Clocks */	
#define MCF5307_DRAMC_DCR_RRA_3			(0x0800)		/* Refresh RAS Asserted 3 Clocks */	
#define MCF5307_DRAMC_DCR_RRA_4			(0x1000)		/* Refresh RAS Asserted 4 Clocks */	
#define MCF5307_DRAMC_DCR_RRA_5			(0x1800)		/* Refresh RAS Asserted 5 Clocks */	
#define MCF5307_DRAMC_DCR_RRP_1			(0x0000)		/* Refresh RAS Precharged 3 Clks */	
#define MCF5307_DRAMC_DCR_RRP_2			(0x0200)		/* Refresh RAS Precharged 3 Clks */	
#define MCF5307_DRAMC_DCR_RRP_3			(0x0400)		/* Refresh RAS Precharged 3 Clks */	
#define MCF5307_DRAMC_DCR_RRP_4			(0x0600)		/* Refresh RAS Precharged 3 Clks */	

#define MCF5307_DRAMC_DACR_CAS_1		(0x00000000)	/* CAS Active 1 Clock */	 	
#define MCF5307_DRAMC_DACR_CAS_2		(0x00001000)	/* CAS Active 2 Clocks */	 	
#define MCF5307_DRAMC_DACR_CAS_3		(0x00002000)	/* CAS Active 3 Clocks */	 	
#define MCF5307_DRAMC_DACR_CAS_4		(0x00003000)	/* CAS Active 4 Clocks */	 	
#define MCF5307_DRAMC_DACR_RP_1			(0x00000000)	/* RAS Precharge 1 Clock */ 	
#define MCF5307_DRAMC_DACR_RP_2			(0x00000400)	/* RAS Precharge 2 Clocks */	
#define MCF5307_DRAMC_DACR_RP_3			(0x00000800)	/* RAS Precharge 3 Clocks */
#define MCF5307_DRAMC_DACR_RP_4			(0x00000C00)	/* RAS Precharge 4 Clocks */	
#define MCF5307_DRAMC_DACR_RNCN			(0x00000200)	/* RAS Negate to CAS Negate	 */
#define MCF5307_DRAMC_DACR_RCD_1		(0x00000000)	/* 1 Clock Between RAS and CAS */	
#define MCF5307_DRAMC_DACR_RCD_2		(0x00000100)	/* 2 Clocks Between RAS and CAS */
#define MCF5307_DRAMC_DACR_EDO			(0x00000040)	/* Extended Data Out */
#define MCF5307_DRAMC_DACR_PM_OFF		(0x00000000)	/* No Page Mode */	
#define MCF5307_DRAMC_DACR_PM_BURST		(0x00000004)	/* Page Mode on Burst Only */
#define MCF5307_DRAMC_DACR_PM_ON		(0x0000000C)	/* Continuous Page Mode */

/*	Controls used only by Synchronous DRAM */
#define MCF5307_DRAMC_DCR_COC		(0x1000) /* Command on Clock Enable */
#define MCF5307_DRAMC_DCR_IS		(0x0800) /* Initiate Self Refresh Command */
#define MCF5307_DRAMC_DCR_RTIM_3	(0x0000) /* 3 Clocks Between REF and ACTV Cmds */
#define MCF5307_DRAMC_DCR_RTIM_6	(0x0200) /* 6 Clocks Between REF and ACTV Cmds */
#define MCF5307_DRAMC_DCR_RTIM_9	(0x0400) /* 9 Clocks Between REF and ACTV Cmds */

#define MCF5307_DRAMC_DACR_CASL_1	(0x00000000) /* 1 Clock From CAS to Data */
#define MCF5307_DRAMC_DACR_CASL_2	(0x00001000) /* 2 Clock From CAS to Data */	
#define MCF5307_DRAMC_DACR_CASL_3	(0x00002000) /* 3 Clock From CAS to Data */	
#define MCF5307_DRAMC_DACR_CBM(a)	(((a)&0x00000007)<<8) /* Command and Bank Mux */
#define MCF5307_DRAMC_DACR_IMRS		(0x00000040) /* Initiate Mode Register Set Cmd */
#define MCF5307_DRAMC_DACR_IP		(0x00000008) /* Initiate Precharge All Command */
#define MCF5307_DRAMC_DACR_PM		(0x00000004) /* Continuous Page	Mode */

/**********************************************************************/
/*	UART Module, UART												  */
/**********************************************************************/

#define MCF5307_UART1_UMR		(0x01C0)
#define MCF5307_UART1_USR		(0x01C4)
#define MCF5307_UART1_UCSR		(0x01C4)
#define MCF5307_UART1_UCR		(0x01C8)
#define MCF5307_UART1_URB		(0x01CC)
#define MCF5307_UART1_UTB		(0x01CC)
#define MCF5307_UART1_UIPCR		(0x01D0)
#define MCF5307_UART1_UACR		(0x01D0)
#define MCF5307_UART1_UISR		(0x01D4)
#define MCF5307_UART1_UIMR		(0x01D4)
#define MCF5307_UART1_UBG1		(0x01D8)
#define MCF5307_UART1_UBG2		(0x01DC)
#define MCF5307_UART1_UIVR		(0x01F0)
#define MCF5307_UART1_UIP		(0x01F4)
#define MCF5307_UART1_UOP1		(0x01F8)
#define MCF5307_UART1_UOP0		(0x01FC)

#define MCF5307_UART2_UMR		(0x0200)
#define MCF5307_UART2_USR		(0x0204)
#define MCF5307_UART2_UCSR		(0x0204)
#define MCF5307_UART2_UCR		(0x0208)
#define MCF5307_UART2_URB		(0x020C)
#define MCF5307_UART2_UTB		(0x020C)
#define MCF5307_UART2_UIPCR		(0x0210)
#define MCF5307_UART2_UACR		(0x0210)
#define MCF5307_UART2_UISR		(0x0214)
#define MCF5307_UART2_UIMR		(0x0214)
#define MCF5307_UART2_UBG1		(0x0218)
#define MCF5307_UART2_UBG2		(0x021C)
#define MCF5307_UART2_UIVR		(0x0230)
#define MCF5307_UART2_UIP		(0x0234)
#define MCF5307_UART2_UOP1		(0x0238)
#define MCF5307_UART2_UOP0		(0x023C)

/* Read access macros for general use */
#define MCF5307_RD_UART1_UMR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UMR,8)
#define MCF5307_RD_UART1_USR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_USR,8)
#define MCF5307_RD_UART1_URB(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_URB,8)
#define MCF5307_RD_UART1_UIPCR(IMMP)	Mcf5307_iord(IMMP,MCF5307_UART1_UIPCR,8)
#define MCF5307_RD_UART1_UISR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UISR,8)
#define MCF5307_RD_UART1_UBG1(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UBG1,8)
#define MCF5307_RD_UART1_UBG2(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UBG2,8)
#define MCF5307_RD_UART1_UIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UIVR,8)
#define MCF5307_RD_UART1_UIP(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART1_UIP,8)

#define MCF5307_RD_UART2_UMR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UMR,8)
#define MCF5307_RD_UART2_USR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_USR,8)
#define MCF5307_RD_UART2_URB(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_URB,8)
#define MCF5307_RD_UART2_UIPCR(IMMP)	Mcf5307_iord(IMMP,MCF5307_UART2_UIPCR,8)
#define MCF5307_RD_UART2_UISR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UISR,8)
#define MCF5307_RD_UART2_UBG1(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UBG1,8)
#define MCF5307_RD_UART2_UBG2(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UBG2,8)
#define MCF5307_RD_UART2_UIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UIVR,8)
#define MCF5307_RD_UART2_UIP(IMMP)		Mcf5307_iord(IMMP,MCF5307_UART2_UIP,8)

/* Write access macros for general use */
#define MCF5307_WR_UART1_UMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UMR,8,DATA)
#define MCF5307_WR_UART1_UCSR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UCSR,8,DATA)
#define MCF5307_WR_UART1_UCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UCR,8,DATA)
#define MCF5307_WR_UART1_UTB(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UTB,8,DATA)
#define MCF5307_WR_UART1_UACR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UACR,8,DATA)
#define MCF5307_WR_UART1_UIMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UIMR,8,DATA)
#define MCF5307_WR_UART1_UBG1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UBG1,8,DATA)
#define MCF5307_WR_UART1_UBG2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UBG2,8,DATA)
#define MCF5307_WR_UART1_UIVR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UIVR,8,DATA)
#define MCF5307_WR_UART1_UOP1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UOP1,8,DATA)
#define MCF5307_WR_UART1_UOP0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART1_UOP0,8,DATA)

#define MCF5307_WR_UART2_UMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UMR,8,DATA)
#define MCF5307_WR_UART2_UCSR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UCSR,8,DATA)
#define MCF5307_WR_UART2_UCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UCR,8,DATA)
#define MCF5307_WR_UART2_UTB(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UTB,8,DATA)
#define MCF5307_WR_UART2_UACR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UACR,8,DATA)
#define MCF5307_WR_UART2_UIMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UIMR,8,DATA)
#define MCF5307_WR_UART2_UBG1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UBG1,8,DATA)
#define MCF5307_WR_UART2_UBG2(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UBG2,8,DATA)
#define MCF5307_WR_UART2_UIVR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UIVR,8,DATA)
#define MCF5307_WR_UART2_UOP1(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UOP1,8,DATA)
#define MCF5307_WR_UART2_UOP0(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_UART2_UOP0,8,DATA)

#if 1
typedef volatile struct
{
	uint8	reserved1[0x1C0];
	uint8	UMR;
	uint8	reserved2;
	uint8	reserved3;
	uint8	reserved4;
	uint8	USR;
	uint8	reserved5;
	uint8	reserved6;
	uint8	reserved7;
	uint8	UCR;
	uint8	reserved8;
	uint8	reserved9;
	uint8	reserved10;
	uint8	UBUF;
	uint8	reserved11;
	uint8	reserved12;
	uint8	reserved13;
	uint8	UACR;
	uint8	reserved14;
	uint8	reserved15;
	uint8	reserved16;
	uint8	UIR;
	uint8	reserved17;
	uint8	reserved18;
	uint8	reserved19;
	uint8	UBG1;
	uint8	reserved20;
	uint8	reserved21;
	uint8	reserved22;
	uint8	UBG2;
	uint32	reserved23[4];
	uint8	UIVR;
	uint8	reserved24;
	uint8	reserved25;
	uint8	reserved26;
	uint8	UIP;
	uint8	reserved27;
	uint8	reserved28;
	uint8	reserved29;
	uint8	UOP1;
	uint8	reserved30;
	uint8	reserved31;
	uint8	reserved32;
	uint8	UOP0;
} MCF5307_UART1;

typedef volatile struct
{
	uint8	reserved1[0x200];
	uint8	UMR;
	uint8	reserved2;
	uint8	reserved3;
	uint8	reserved4;
	uint8	USR;
	uint8	reserved5;
	uint8	reserved6;
	uint8	reserved7;
	uint8	UCR;
	uint8	reserved8;
	uint8	reserved9;
	uint8	reserved10;
	uint8	UBUF;
	uint8	reserved11;
	uint8	reserved12;
	uint8	reserved13;
	uint8	UACR;
	uint8	reserved14;
	uint8	reserved15;
	uint8	reserved16;
	uint8	UIR;
	uint8	reserved17;
	uint8	reserved18;
	uint8	reserved19;
	uint8	UBG1;
	uint8	reserved20;
	uint8	reserved21;
	uint8	reserved22;
	uint8	UBG2;
	uint32	reserved23[4];
	uint8	UIVR;
	uint8	reserved24;
	uint8	reserved25;
	uint8	reserved26;
	uint8	UIP;
	uint8	reserved27;
	uint8	reserved28;
	uint8	reserved29;
	uint8	UOP1;
	uint8	reserved30;
	uint8	reserved31;
	uint8	reserved32;
	uint8	UOP0;
} MCF5307_UART2;
#endif

#define MCF5307_UART_UMR1_RXRTS				(0x80)
#define MCF5307_UART_UMR1_RXIRQ				(0x40)
#define MCF5307_UART_UMR1_ERR				(0x20)
#define MCF5307_UART_UMR1_PM1				(0x10)
#define MCF5307_UART_UMR1_PM0				(0x08)
#define MCF5307_UART_UMR1_PMT				(0x04)
#define MCF5307_UART_UMR1_BC1				(0x02)
#define MCF5307_UART_UMR1_BC0				(0x01)
#define MCF5307_UART_UMR1_PM_MULTI_ADDR		(0x1C)
#define MCF5307_UART_UMR1_PM_MULTI_DATA		(0x18)
#define MCF5307_UART_UMR1_PM_NONE			(0x10)
#define MCF5307_UART_UMR1_PM_FORCE_HI		(0x0C)
#define MCF5307_UART_UMR1_PM_FORCE_LO		(0x08)
#define MCF5307_UART_UMR1_PM_ODD			(0x04)
#define MCF5307_UART_UMR1_PM_EVEN			(0x00)
#define MCF5307_UART_UMR1_BC_5				(0x00)
#define MCF5307_UART_UMR1_BC_6				(0x01)
#define MCF5307_UART_UMR1_BC_7				(0x02)
#define MCF5307_UART_UMR1_BC_8				(0x03)

#define MCF5307_UART_UMR2_CM1				(0x80)
#define MCF5307_UART_UMR2_CM0				(0x40)
#define MCF5307_UART_UMR2_TXRTS				(0x20)
#define MCF5307_UART_UMR2_TXCTS				(0x10)
#define MCF5307_UART_UMR2_SB3				(0x08)
#define MCF5307_UART_UMR2_SB2				(0x04)
#define MCF5307_UART_UMR2_SB1				(0x02)
#define MCF5307_UART_UMR2_SB0				(0x01)
#define MCF5307_UART_UMR2_CM_NORMAL			(0x00)
#define MCF5307_UART_UMR2_CM_ECHO			(0x40)
#define MCF5307_UART_UMR2_CM_LOCAL_LOOP		(0x80)
#define MCF5307_UART_UMR2_CM_REMOTE_LOOP	(0xC0)
#define MCF5307_UART_UMR2_STOP_BITS_1		(0x07)
#define MCF5307_UART_UMR2_STOP_BITS_15		(0x08)
#define MCF5307_UART_UMR2_STOP_BITS_2		(0x0F)

#define MCF5307_UART_USR_RB					(0x80)
#define MCF5307_UART_USR_FE					(0x40)
#define MCF5307_UART_USR_PE					(0x20)
#define MCF5307_UART_USR_OE					(0x10)
#define MCF5307_UART_USR_TXEMP				(0x08)
#define MCF5307_UART_USR_TXRDY				(0x04)
#define MCF5307_UART_USR_FFULL				(0x02)
#define MCF5307_UART_USR_RXRDY				(0x01)

#define MCF5307_UART_UCSR_RCS3				(0x80)
#define MCF5307_UART_UCSR_RCS2				(0x40)
#define MCF5307_UART_UCSR_RCS1				(0x20)
#define MCF5307_UART_UCSR_RCS0				(0x10)
#define MCF5307_UART_UCSR_TCS3				(0x08)
#define MCF5307_UART_UCSR_TCS2				(0x04)
#define MCF5307_UART_UCSR_TCS1				(0x02)
#define MCF5307_UART_UCSR_TCS0				(0x01)
#define MCF5307_UART_UCSR_RX_TIMER			(0xD0)
#define MCF5307_UART_UCSR_RX_16EXT			(0xE0)
#define MCF5307_UART_UCSR_RX_1EXT			(0xF0)
#define MCF5307_UART_UCSR_TX_TIMER			(0x0D)
#define MCF5307_UART_UCSR_TX_16EXT			(0x0E)
#define MCF5307_UART_UCSR_TX_1EXT			(0x0F)

#define MCF5307_UART_UCR_MISC2				(0x40)
#define MCF5307_UART_UCR_MISC1				(0x20)
#define MCF5307_UART_UCR_MISC0				(0x10)
#define MCF5307_UART_UCR_TC1				(0x08)
#define MCF5307_UART_UCR_TC0				(0x04)
#define MCF5307_UART_UCR_RC1				(0x02)
#define MCF5307_UART_UCR_RC0				(0x01)
#define MCF5307_UART_UCR_NONE				(0x00)
#define MCF5307_UART_UCR_STOP_BREAK			(0x70)
#define MCF5307_UART_UCR_START_BREAK		(0x60)
#define MCF5307_UART_UCR_RESET_BKCHGINT		(0x50)
#define MCF5307_UART_UCR_RESET_ERROR		(0x40)
#define MCF5307_UART_UCR_RESET_TX			(0x30)
#define MCF5307_UART_UCR_RESET_RX			(0x20)
#define MCF5307_UART_UCR_RESET_MR			(0x10)
#define MCF5307_UART_UCR_TX_DISABLED		(0x08)
#define MCF5307_UART_UCR_TX_ENABLED			(0x04)
#define MCF5307_UART_UCR_RX_DISABLED		(0x02)
#define MCF5307_UART_UCR_RX_ENABLED			(0x01)

#define MCF5307_UART_UIPCR_COS				(0x10)
#define MCF5307_UART_UIPCR_CTS				(0x01)

#define MCF5307_UART_UACR_IEC				(0x01)

#define MCF5307_UART_UISR_COS				(0x80)
#define MCF5307_UART_UISR_DB				(0x04)
#define MCF5307_UART_UISR_RXRDY				(0x02)
#define MCF5307_UART_UISR_TXRDY				(0x01)

#define MCF5307_UART_UIMR_COS				(0x80)
#define MCF5307_UART_UIMR_DB				(0x04)
#define MCF5307_UART_UIMR_FFULL				(0x02)
#define MCF5307_UART_UIMR_TXRDY				(0x01)

#define MCF5307_UART_UIP_CTS				(0x01)

#define MCF5307_UART_UOP1_RTS				(0x01)

#define MCF5307_UART_UOP0_RTS				(0x01)

/***********************************************************************/
/*	M-BUS Registers													   */
/***********************************************************************/

#define MCF5307_MBUS_MADR		(0x0280)
#define MCF5307_MBUS_MFDR		(0x0284)
#define MCF5307_MBUS_MBCR		(0x0288)
#define MCF5307_MBUS_MBSR		(0x028C)
#define MCF5307_MBUS_MBDR		(0x0290)

/* Read access macros for general use */
#define MCF5307_RD_MBUS_MADR(IMMP)		Mcf5307_iord(IMMP,MCF5307_MBUS_MADR,8)
#define MCF5307_RD_MBUS_MFDR(IMMP)		Mcf5307_iord(IMMP,MCF5307_MBUS_MFDR,8)
#define MCF5307_RD_MBUS_MBCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_MBUS_MBCR,8)
#define MCF5307_RD_MBUS_MBSR(IMMP)		Mcf5307_iord(IMMP,MCF5307_MBUS_MBSR,8)
#define MCF5307_RD_MBUS_MBDR(IMMP)		Mcf5307_iord(IMMP,MCF5307_MBUS_MBDR,8)

/* Write access macros for general use */
#define MCF5307_WR_MBUS_MADR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_MBUS_MADR,8,DATA)
#define MCF5307_WR_MBUS_MFDR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_MBUS_MFDR,8,DATA)
#define MCF5307_WR_MBUS_MBCR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_MBUS_MBCR,8,DATA)
#define MCF5307_WR_MBUS_MBSR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_MBUS_MBSR,8,DATA)
#define MCF5307_WR_MBUS_MBDR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_MBUS_MBDR,8,DATA)

typedef volatile struct
{
	uint32	reserved0[0xa0];
	uint8	MADR;		/* M-Bus Address Register */		
	uint8	reserved1;
	uint16	reserved2;
	uint8	MFDR;		/* M-Bus Frequency Divider Register */	
	uint8	reserved3;
	uint16	reserved4;
	uint8	MBCR;		/* M-Bus Control Register */		
	uint8	reserved5;
	uint16	reserved6;
	uint8	MBSR;		/* M-Bus Status Register */		
	uint8	reserved7;
	uint16	reserved8;
	uint8	MBDR;		/* M-Bus Data I/O Register */		
} MCF5307_MBUS;

#define MCF5307_MBUS_MADR_ADDR(a)	(((a)&0xFE)<<0x01) /* Slave Address */	

#define MCF5307_MBUS_MFDR_MBC(a)	((a)&0x3F)	/* M-Bus Clock Rate */	

#define MCF5307_MBUS_MBCR_MEN		(0x80)	/* M-Bus Enable */			
#define MCF5307_MBUS_MBCR_MIEN		(0x40)	/* M-Bus Interrupt Enable */	
#define MCF5307_MBUS_MBCR_MSTA		(0x20)	/* Master/Slave Mode Select Bit */	
#define MCF5307_MBUS_MBCR_MTX		(0x10)	/* Transmit/Rcv Mode Select Bit */	
#define MCF5307_MBUS_MBCR_TXAK		(0x08)	/* Transmit Acknowledge Enable */	
#define MCF5307_MBUS_MBCR_RSTA		(0x04)	/* Repeat Start */			

#define MCF5307_MBUS_MBSR_MCF		(0x80)	/* Data Transfer Complete */	
#define MCF5307_MBUS_MBSR_MAAS		(0x40)	/* Addressed as a Slave */		
#define MCF5307_MBUS_MBSR_MBB		(0x20)	/* Bus Busy */			
#define MCF5307_MBUS_MBSR_MAL		(0x10)	/* Arbitration Lost */		
#define MCF5307_MBUS_MBSR_SRW		(0x04)	/* Slave Transmit */		
#define MCF5307_MBUS_MBSR_MIF		(0x02)	/* M-Bus Interrupt */		
#define MCF5307_MBUS_MBSR_RXAK		(0x01)	/* No Acknowledge Received */	

/***********************************************************************/
/*  Timer Registers													   */
/***********************************************************************/

#define MCF5307_TIMER1_TMR		(0x0140)
#define MCF5307_TIMER1_TRR		(0x0144)
#define MCF5307_TIMER1_TCR		(0x0148)
#define MCF5307_TIMER1_TCN		(0x014C)
#define MCF5307_TIMER1_TER		(0x0151)

#define MCF5307_TIMER2_TMR		(0x0180)
#define MCF5307_TIMER2_TRR		(0x0184)
#define MCF5307_TIMER2_TCR		(0x0188)
#define MCF5307_TIMER2_TCN		(0x018C)
#define MCF5307_TIMER2_TER		(0x0191)

/* Read access macros for general use */
#define MCF5307_RD_TIMER1_TMR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER1_TMR,16)
#define MCF5307_RD_TIMER1_TRR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER1_TRR,16)
#define MCF5307_RD_TIMER1_TCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER1_TCR,16)
#define MCF5307_RD_TIMER1_TCN(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER1_TCN,16)
#define MCF5307_RD_TIMER1_TER(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER1_TER,8)

#define MCF5307_RD_TIMER2_TMR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER2_TMR,16)
#define MCF5307_RD_TIMER2_TRR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER2_TRR,16)
#define MCF5307_RD_TIMER2_TCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER2_TCR,16)
#define MCF5307_RD_TIMER2_TCN(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER2_TCN,16)
#define MCF5307_RD_TIMER2_TER(IMMP)		Mcf5307_iord(IMMP,MCF5307_TIMER2_TER,8)

/* Write access macros for general use */
#define MCF5307_WR_TIMER1_TMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER1_TMR,16,DATA)
#define MCF5307_WR_TIMER1_TRR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER1_TRR,16,DATA)
#define MCF5307_WR_TIMER1_TCN(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER1_TCN,16,DATA)
#define MCF5307_WR_TIMER1_TER(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER1_TER,8,DATA)

#define MCF5307_WR_TIMER2_TMR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER2_TMR,16,DATA)
#define MCF5307_WR_TIMER2_TRR(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER2_TRR,16,DATA)
#define MCF5307_WR_TIMER2_TCN(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER2_TCN,16,DATA)
#define MCF5307_WR_TIMER2_TER(IMMP,DATA)	\
	Mcf5307_iowr(IMMP,MCF5307_TIMER2_TER,8,DATA)

typedef volatile struct
{
	uint32	reserved0[0x50];
	uint16	TMR1;		/* Timer 1 Mode Register */	
	uint16	reserved1;
	uint16	TRR1;		/* Timer 1 Reference Register */	
	uint16	reserved2;
	uint16	TCR1;		/* Timer 1 Capture Register */	
	uint16	reserved3;
	uint16	TCN1;		/* Timer 1 Counter */		
	uint16	reserved4;
	uint8	reserved5;
	uint8	TER1;		/* Timer 1 Event Register */	
	uint16	reserved6;
	uint32	reserved7[0xb];
	uint16	TMR2;		/* Timer 2 Mode Register */	
	uint16	reserved8;
	uint16	TRR2;		/* Timer 2 Reference Register */	
	uint16	reserved9;
	uint16	TCR2;		/* Timer 2 Capture Register */	
	uint16	reserved10;
	uint16	TCN2;		/* Timer 2 Counter */		
	uint16	reserved11;
	uint8	reserved12;
	uint8	TER2;		/* Timer 2 Event Register */	
} MCF5307_TIMER;

#define MCF5307_TIMER_TMR_PS(a)		(((a)&0x00FF)<<8) /* Prescaler Value */		
#define MCF5307_TIMER_TMR_CE_ANY	(0x00C0) /* Capture on Any Edge */			
#define MCF5307_TIMER_TMR_CE_FALL	(0x0080) /* Capture on Falling Edge */		
#define MCF5307_TIMER_TMR_CE_RISE	(0x0040) /* Capture on Rising Edge */		
#define MCF5307_TIMER_TMR_CE_NONE	(0x0000) /* Disable Capture Event */		
#define MCF5307_TIMER_TMR_OM		(0x0020) /* Output Mode */				
#define MCF5307_TIMER_TMR_ORI		(0x0010) /* Output Reference Interrupt Enable */	
#define MCF5307_TIMER_TMR_FRR		(0x0008) /* Restart After Reference Value */	
#define MCF5307_TIMER_TMR_CLK_TIN	(0x0006) /* TIN is Input Clock Source */		
#define MCF5307_TIMER_TMR_CLK_DIV16	(0x0004) /* Sys Clk / 16 is Input Clock Source */	
#define MCF5307_TIMER_TMR_CLK_MSCLK	(0x0002) /* Sys Clk is Input Clock Source */	
#define MCF5307_TIMER_TMR_CLK_STOP	(0x0000) /* Stop Count */				
#define MCF5307_TIMER_TMR_RST		(0x0001) /* Enable Timer */			

#define MCF5307_TIMER_TER_REF		(0x02)	/* Output Reference Event */		
#define MCF5307_TIMER_TER_CAP		(0x01)	/* Capture Event */			

/***********************************************************************/
/*  DMA Registers													   */
/***********************************************************************/

#define MCF5307_DMA0_SAR		(0x0300)
#define MCF5307_DMA0_DAR		(0x0304)
#define MCF5307_DMA0_DCR		(0x0308)
#define MCF5307_DMA0_BCR		(0x030C)
#define MCF5307_DMA0_DSR		(0x0310)
#define MCF5307_DMA0_DIVR		(0x0314)

#define MCF5307_DMA1_SAR		(0x0340)
#define MCF5307_DMA1_DAR		(0x0344)
#define MCF5307_DMA1_DCR		(0x0348)
#define MCF5307_DMA1_BCR		(0x034C)
#define MCF5307_DMA1_DSR		(0x0350)
#define MCF5307_DMA1_DIVR		(0x0354)

#define MCF5307_DMA2_SAR		(0x0380)
#define MCF5307_DMA2_DAR		(0x0384)
#define MCF5307_DMA2_DCR		(0x0388)
#define MCF5307_DMA2_BCR		(0x038C)
#define MCF5307_DMA2_DSR		(0x0390)
#define MCF5307_DMA2_DIVR		(0x0394)

#define MCF5307_DMA3_SAR		(0x03C0)
#define MCF5307_DMA3_DAR		(0x03C4)
#define MCF5307_DMA3_DCR		(0x03C8)
#define MCF5307_DMA3_BCR		(0x03CC)
#define MCF5307_DMA3_DSR		(0x03D0)
#define MCF5307_DMA3_DIVR		(0x03D4)

/* Read access macros for general use */
#define MCF5307_RD_DMA0_SAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_SAR,32)
#define MCF5307_RD_DMA0_DAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_DAR,32)
#define MCF5307_RD_DMA0_DCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_DCR,16)
#define MCF5307_RD_DMA0_BCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_BCR,16)
#define MCF5307_RD_DMA0_BCR24(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_BCR,32)
#define MCF5307_RD_DMA0_DSR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_DSR,8)
#define MCF5307_RD_DMA0_DIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA0_DIVR,8)

#define MCF5307_RD_DMA1_SAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_SAR,32)
#define MCF5307_RD_DMA1_DAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_DAR,32)
#define MCF5307_RD_DMA1_DCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_DCR,16)
#define MCF5307_RD_DMA1_BCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_BCR,16)
#define MCF5307_RD_DMA1_BCR24(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_BCR,32)
#define MCF5307_RD_DMA1_DSR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_DSR,8)
#define MCF5307_RD_DMA1_DIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA1_DIVR,8)

#define MCF5307_RD_DMA2_SAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_SAR,32)
#define MCF5307_RD_DMA2_DAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_DAR,32)
#define MCF5307_RD_DMA2_DCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_DCR,16)
#define MCF5307_RD_DMA2_BCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_BCR,16)
#define MCF5307_RD_DMA2_BCR24(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_BCR,32)
#define MCF5307_RD_DMA2_DSR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_DSR,8)
#define MCF5307_RD_DMA2_DIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA2_DIVR,8)

#define MCF5307_RD_DMA3_SAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_SAR,32)
#define MCF5307_RD_DMA3_DAR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_DAR,32)
#define MCF5307_RD_DMA3_DCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_DCR,16)
#define MCF5307_RD_DMA3_BCR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_BCR,16)
#define MCF5307_RD_DMA3_BCR24(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_BCR,32)
#define MCF5307_RD_DMA3_DSR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_DSR,8)
#define MCF5307_RD_DMA3_DIVR(IMMP)		Mcf5307_iord(IMMP,MCF5307_DMA3_DIVR,8)

/* Write access macros for general use */
#define MCF5307_WR_DMA0_SAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_SAR,32,DATA)
#define MCF5307_WR_DMA0_DAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_DAR,32,DATA)
#define MCF5307_WR_DMA0_DCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_DCR,16,DATA)
#define MCF5307_WR_DMA0_BCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_BCR,16,DATA)
#define MCF5307_WR_DMA0_BCR24(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_BCR,32,DATA)
#define MCF5307_WR_DMA0_DSR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_DSR,8,DATA)
#define MCF5307_WR_DMA0_DIVR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA0_DIVR,8,DATA)

#define MCF5307_WR_DMA1_SAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_SAR,32,DATA)
#define MCF5307_WR_DMA1_DAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_DAR,32,DATA)
#define MCF5307_WR_DMA1_DCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_DCR,16,DATA)
#define MCF5307_WR_DMA1_BCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_BCR,16,DATA)
#define MCF5307_WR_DMA1_BCR24(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_BCR,32,DATA)
#define MCF5307_WR_DMA1_DSR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_DSR,8,DATA)
#define MCF5307_WR_DMA1_DIVR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA1_DIVR,8,DATA)

#define MCF5307_WR_DMA2_SAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_SAR,32,DATA)
#define MCF5307_WR_DMA2_DAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_DAR,32,DATA)
#define MCF5307_WR_DMA2_DCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_DCR,16,DATA)
#define MCF5307_WR_DMA2_BCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_BCR,16,DATA)
#define MCF5307_WR_DMA2_BCR24(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_BCR,32,DATA)
#define MCF5307_WR_DMA2_DSR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_DSR,8,DATA)
#define MCF5307_WR_DMA2_DIVR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA2_DIVR,8,DATA)

#define MCF5307_WR_DMA3_SAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_SAR,32,DATA)
#define MCF5307_WR_DMA3_DAR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_DAR,32,DATA)
#define MCF5307_WR_DMA3_DCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_DCR,16,DATA)
#define MCF5307_WR_DMA3_BCR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_BCR,16,DATA)
#define MCF5307_WR_DMA3_BCR24(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_BCR,32,DATA)
#define MCF5307_WR_DMA3_DSR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_DSR,8,DATA)
#define MCF5307_WR_DMA3_DIVR(IMMP,DATA)	Mcf5307_iowr(IMMP,MCF5307_DMA3_DIVR,8,DATA)

typedef volatile struct
{
	uint32	reserved0[0xc0];
	uint32	SAR0;		/* DMA 0 Source Address Register */	
	uint32	DAR0;		/* DMA 0 Destination Address Register */	
	uint16	DCR0;		/* DMA 0 Control Register */		
	uint16	reserved1;
	uint16	BCR0;		/* DMA 0 Byte Count Register */		
	uint16	reserved2;
	uint8	DSR0;		/* DMA 0 Status Register */		
	uint8	reserved3;
	uint16	reserved4;
	uint8	DIVR0;		/* DMA 0 Interrupt Vector Register */	
	uint8	reserved5;
	uint16	reserved6;
	uint32	reserved7[0xa];
	uint32	SAR1;		/* DMA 1 Source Address Register */	
	uint32	DAR1;		/* DMA 1 Destination Address Register */	
	uint16	DCR1;		/* DMA 1 Control Register */		
	uint16	reserved8;
	uint16	BCR1;		/* DMA 1 Byte Count Register */		
	uint16	reserved9;
	uint8	DSR1;		/* DMA 1 Status Register */		
	uint8	reserved10;
	uint16	reserved11;
	uint8	DIVR1;		/* DMA 1 Interrupt Vector Register */	
	uint8	reserved12;
	uint16	reserved13;
	uint32	reserved14[0xa];
	uint32	SAR2;		/* DMA 2 Source Address Register */	
	uint32	DAR2;		/* DMA 2 Destination Address Register */	
	uint16	DCR2;		/* DMA 2 Control Register */		
	uint16	reserved15;
	uint16	BCR2;		/* DMA 2 Byte Count Register */		
	uint16	reserved16;
	uint8	DSR2;		/* DMA 2 Status Register */		
	uint8	reserved17;
	uint16	reserved18;
	uint8	DIVR2;		/* DMA 2 Interrupt Vector Register */	
	uint8	reserved19;
	uint16	reserved20;
	uint32	reserved21[0xa];
	uint32	SAR3;		/* DMA 3 Source Address Register */	
	uint32	DAR3;		/* DMA 3 Destination Address Register */	
	uint16	DCR3;		/* DMA 3 Control Register */		
	uint16	reserved22;
	uint16	BCR3;		/* DMA 3 Byte Count Register */		
	uint16	reserved23;
	uint8	DSR3;		/* DMA 3 Status Register */		
	uint8	reserved24;
	uint16	reserved25;
	uint8	DIVR3;		/* DMA 3 Interrupt Vector Register */	
} MCF5307_DMA;

#define MCF5307_DMA_DCR_INT			(0x8000)	/* Interrupt on Completion */	
#define MCF5307_DMA_DCR_EEXT		(0x4000)	/* Enable External Request */	
#define MCF5307_DMA_DCR_CS			(0x2000)	/* Cycle Steal */			
#define MCF5307_DMA_DCR_AA			(0x1000)	/* Auto Align */			
#define MCF5307_DMA_DCR_BWC_DMA		(0x0000)	/* Bandwidth: DMA Priority */	
#define MCF5307_DMA_DCR_BWC_512		(0x0200)	/* Bandwidth:   512 Bytes */	
#define MCF5307_DMA_DCR_BWC_1024	(0x0400)	/* Bandwidth:  1024 Bytes */	
#define MCF5307_DMA_DCR_BWC_2048	(0x0600)	/* Bandwidth:  2048 Bytes */	
#define MCF5307_DMA_DCR_BWC_4096	(0x0800)	/* Bandwidth:  4096 Bytes */	
#define MCF5307_DMA_DCR_BWC_8192	(0x0a00)	/* Bandwidth:  8192 Bytes */	
#define MCF5307_DMA_DCR_BWC_16384	(0x0c00)	/* Bandwidth: 16384 Bytes */	
#define MCF5307_DMA_DCR_BWC_32768	(0x0e00)	/* Bandwidth: 32768 Bytes */	
#define MCF5307_DMA_DCR_SAA			(0x0100)	/* Single Address Access */	
#define MCF5307_DMA_DCR_SRW			(0x0080)	/* Forces MRW Signal High */	
#define	MCF5307_DMA_DCR_SINC		(0x0040)	/* Source Increment */		
#define MCF5307_DMA_DCR_SSIZE_LONG	(0x0000)	/* Source Size:  Longword */	
#define MCF5307_DMA_DCR_SSIZE_BYTE	(0x0010)	/* Source Size:  Byte */		
#define MCF5307_DMA_DCR_SSIZE_WORD	(0x0020)	/* Source Size:  Word */		
#define MCF5307_DMA_DCR_SSIZE_LINE	(0x0030)	/* Source Size:  Line */		
#define MCF5307_DMA_DCR_DINC		(0x0008)	/* Destination Increment */	
#define MCF5307_DMA_DCR_DSIZE_LONG	(0x0000)	/* Destination Size:  Longword */	
#define MCF5307_DMA_DCR_DSIZE_BYTE	(0x0002)	/* Destination Size:  Byte */	
#define MCF5307_DMA_DCR_DSIZE_WORD	(0x0004)	/* Destination Size:  Word */	
#define MCF5307_DMA_DCR_START		(0x0001)	/* Start Transfer */		

#define MCF5307_DMA_DSR_CE			(0x40)		/* Configuration Error */		
#define MCF5307_DMA_DSR_BES			(0x20)		/* Bus Error on Source */		
#define MCF5307_DMA_DSR_BED			(0x10)		/* Bus Error on Destination */
#define MCF5307_DMA_DSR_REQ			(0x04)		/* Request */			
#define MCF5307_DMA_DSR_BSY			(0x02)		/* Busy */				
#define MCF5307_DMA_DSR_DONE		(0x01)		/* Transaction Done */		


/***********************************************************************/
/*	Here we put the modules together.  								   */
/***********************************************************************/

typedef volatile union
{
	MCF5307_SIM		sim;
	MCF5307_PP		pp;
	MCF5307_UART1	uart1;
	MCF5307_UART2	uart2;
	MCF5307_TIMER	timer;
	MCF5307_CS		cs;
	MCF5307_DRAMC	dramc;
	MCF5307_MBUS	mbus;
	MCF5307_DMA		dma;
} MCF5307_IMM;

#endif	// _CPU_MCF5307_H 
